-- Create database
CREATE DATABASE IF NOT EXISTS candidate_management;
USE candidate_management;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    role ENUM('admin', 'recruiter', 'user') DEFAULT 'user',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Sources table
CREATE TABLE IF NOT EXISTS sources (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Profiles table
CREATE TABLE IF NOT EXISTS profiles (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Candidates table
CREATE TABLE IF NOT EXISTS candidates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    candidate_id VARCHAR(20) UNIQUE NOT NULL,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    location VARCHAR(100) NOT NULL,
    team_name VARCHAR(100) NOT NULL,
    referral_code VARCHAR(50),
    lineup_date DATE NOT NULL,
    source_id INT,
    profile_id INT,
    status ENUM('pending', 'shortlisted', 'selected', 'rejected', 'hold', 'joined', 'rescheduled', 'process_ended', 'offer_declined', 'not_joined') DEFAULT 'pending',
    created_by INT,
    added_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (source_id) REFERENCES sources(id),
    FOREIGN KEY (profile_id) REFERENCES profiles(id),
    FOREIGN KEY (created_by) REFERENCES users(id)
);

-- Interviews table
CREATE TABLE IF NOT EXISTS interviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    candidate_id VARCHAR(20) NOT NULL,
    interview_type VARCHAR(50),
    interview_date DATE,
    interviewer_name VARCHAR(100),
    result VARCHAR(50),
    notes TEXT,
    status VARCHAR(50),
    created_by INT,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (candidate_id) REFERENCES candidates(candidate_id),
    FOREIGN KEY (created_by) REFERENCES users(id)
);

-- Insert default admin user (password: admin123)
INSERT INTO users (username, email, password, role) VALUES 
('admin', 'admin@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin');

-- Insert default sources
INSERT INTO sources (name, description) VALUES 
('Job Portal', 'Candidates from job portals like Naukri, Monster'),
('Referral', 'Employee referrals'),
('Walk-in', 'Walk-in candidates'),
('Social Media', 'LinkedIn, Facebook, etc.'),
('Campus Hiring', 'College campus recruitment');

-- Insert default profiles
INSERT INTO profiles (name, description) VALUES 
('Software Developer', 'Full stack and backend developers'),
('Frontend Developer', 'UI/UX and frontend specialists'),
('Data Analyst', 'Data analysis and reporting roles'),
('Project Manager', 'Project management positions'),
('Quality Assurance', 'Testing and QA roles'),
('DevOps Engineer', 'Infrastructure and deployment'),
('Business Analyst', 'Business requirements and analysis'),
('HR Executive', 'Human resources positions');

-- Create indexes for better performance
CREATE INDEX idx_candidates_status ON candidates(status);
CREATE INDEX idx_candidates_lineup_date ON candidates(lineup_date);
CREATE INDEX idx_candidates_created_by ON candidates(created_by);
CREATE INDEX idx_interviews_candidate_id ON interviews(candidate_id);
CREATE INDEX idx_interviews_created_date ON interviews(created_date);
